// Copyright 2014 Google Inc. All Rights Reserved.

#include "MediaSinkBase.h"

int MediaSinkBase::routeMessage(uint8_t channelId, uint16_t type, const shared_ptr<IoBuffer>& msg) {
    int ret = STATUS_UNEXPECTED_MESSAGE;
    uint8_t* ptr = (uint8_t*)msg->raw() + sizeof(uint16_t);
    size_t len = msg->size() - sizeof(uint16_t);
    uint64_t timestamp = 0;

    switch (type) {
        case MEDIA_MESSAGE_SETUP: {
            Setup req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleSetup(req.type());
            }
            break;
         }
        case MEDIA_MESSAGE_START: {
            Start req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleStart(req);
            }
            break;
        }
        case MEDIA_MESSAGE_STOP: {
            Stop req;
            if (PARSE_PROTO(req, ptr, len)) {
                ret = handleStop(req);
            }
            break;
        }
        case MEDIA_MESSAGE_DATA:
            READ_BE64(ptr, timestamp);
            handleDataAvailable(timestamp, msg, sizeof(uint16_t) + sizeof(uint64_t));
            ret = STATUS_SUCCESS;
            break;
        case MEDIA_MESSAGE_CODEC_CONFIG:
            handleCodecConfig(ptr, len);
            ret = STATUS_SUCCESS;
            break;
        case MEDIA_MESSAGE_VIDEO_FOCUS_REQUEST: {
            VideoFocusRequestNotification req;
            if (PARSE_PROTO(req, ptr, len)) {
                if (handleVideoFocusRequest(req)) {
                    ret = STATUS_SUCCESS;
                }
            }
            break;
        }
        default:
            // size_t is unsigned int with the internal compiler & unsigned long externally.
            LOGE("Failed to dispatch message of type %d size=%lu",
                    type, (unsigned long) msg->size());
    }

    return ret;
}

void MediaSinkBase::sendConfig(int status) {
    Config resp;
    resp.set_status((Config_Status)status);
    resp.set_max_unacked(mMaxUnackedFrames);
    if (mConfigurationPreference.size() == 0) { // preference based on addition time
        for (unsigned int i = 0; i < getNumberOfConfigurations(); i++) {
            resp.add_configuration_indices(i);
        }
    } else {
        for (unsigned int i = 0; i < mConfigurationPreference.size(); i++) {
            resp.add_configuration_indices(mConfigurationPreference[i]);
        }
    }
    IoBuffer buf;
    mRouter->marshallProto(MEDIA_MESSAGE_CONFIG, resp, &buf);
    queueOutgoing(buf.raw(), buf.size());
}

bool MediaSinkBase::resetAllowedConfigurations(uint32_t* configIndices, unsigned int size) {
    unsigned int numberOfConfigs = getNumberOfConfigurations();
    if (size > numberOfConfigs) {
        return false;
    }
    unsigned int maxIndex = numberOfConfigs - 1;
    for (unsigned int i = 0; i < size; i++) {
        if (configIndices[i] > maxIndex) {
            return false;
        }
    }
    mConfigurationPreference.clear();
    for (unsigned int i = 0; i < size; i++) {
        mConfigurationPreference.push_back(configIndices[i]);
    }
    return true;
}

int MediaSinkBase::handleStart(const Start& req) {
    mCurrentSessionId = req.session_id();

    int ret;
    ret = handleMediaConfiguration(req.configuration_index());
    if (ret != STATUS_SUCCESS) {
        return ret;
    }

    playbackStart(mCurrentSessionId);
    return STATUS_SUCCESS;
}

int MediaSinkBase::handleStop(const Stop& req) {
    playbackStop(mCurrentSessionId);
    return STATUS_SUCCESS;
}

bool MediaSinkBase::ackFrames(int32_t sessionId, uint32_t numFrames) {
    Ack ack;
    ack.set_session_id(sessionId);
    ack.set_ack(numFrames);

    IoBuffer buf;
    mRouter->marshallProto(MEDIA_MESSAGE_ACK, ack, &buf);
    return queueOutgoing(buf.raw(), buf.size());
}

